<?php
// Paths to the JSON files
$captionsFile = 'captions.json';
$emojiFile = 'emoji-en-US.json';
$emojiCaptionsFile = 'emoji_captions.json';

// Read the captions.json file
$captionsData = json_decode(file_get_contents($captionsFile), true);
if ($captionsData === null) {
    die('Error reading captions.json');
}

// Read the emoji-en-US.json file
$emojiData = json_decode(file_get_contents($emojiFile), true);
if ($emojiData === null) {
    die('Error reading emoji-en-US.json');
}

// Create a reverse mapping of keywords to emojis
$keywordToEmojiMap = [];
foreach ($emojiData as $emoji => $keywords) {
    foreach ($keywords as $keyword) {
        $keywordToEmojiMap[strtolower($keyword)] = $emoji;
    }
}

// Function to get the emoji for a word and avoid problematic characters (flags, regional indicators)
function getEmojiForWord($word, $keywordToEmojiMap) {
    $wordLower = strtolower($word);
    if (isset($keywordToEmojiMap[$wordLower])) {
        $emoji = $keywordToEmojiMap[$wordLower];
        // Check if the emoji contains regional indicators (flags) and skip them
        if (preg_match('/\p{Regional_Indicator}/u', $emoji)) {
            return ''; // Skip flag emojis
        }
        return $emoji;
    }
    return '';
}

// Generate emoji captions data
$emojiCaptionsData = [];
foreach ($captionsData as $index => $caption) {
    // Split text into words, using only whitespace to ensure exact word matching
    $words = preg_split('/\s+/', $caption['text']);
    $emojiText = implode(' ', array_map(function($word) use ($keywordToEmojiMap) {
        return getEmojiForWord($word, $keywordToEmojiMap);
    }, $words));

    // If no valid emoji, continue to the next caption
    if (trim($emojiText) === '') {
        continue;
    }

    // Adjust start time: add 500 ms (0.5 seconds) without overlapping with previous end time
    $adjustedStartTime = max(0, $caption['startTime'] - 0.5);
    if (isset($emojiCaptionsData[count($emojiCaptionsData) - 1])) {
        $previousEndTime = $emojiCaptionsData[count($emojiCaptionsData) - 1]['endTime'];
        if ($adjustedStartTime < $previousEndTime) {
            $adjustedStartTime = $previousEndTime; // Ensure no overlap with previous emoji
        }
    }

    // Extend the end time by 1000 ms (1 second) unless another emoji starts within this time
    $extendedEndTime = $caption['endTime'] + 1.0;
    if (isset($captionsData[$index + 1])) {
        $nextStartTime = $captionsData[$index + 1]['startTime'];
        if ($nextStartTime < $extendedEndTime) {
            $extendedEndTime = $nextStartTime; // Adjust end time to right before the next emoji
        }
    }

    // Add the emoji caption to the result
    $emojiCaptionsData[] = [
        'text' => trim($emojiText),
        'startTime' => $adjustedStartTime,
        'endTime' => $extendedEndTime
    ];
}

// Write the emoji_captions.json file
file_put_contents($emojiCaptionsFile, json_encode($emojiCaptionsData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
echo 'emoji_captions.json has been generated successfully!';
?>
