const puppeteer = require('puppeteer');
const fs = require('fs');
const ffmpeg = require('fluent-ffmpeg');
const stream = require('stream');
const path = require('path');

(async () => {
    const pidFile = '/tmp/record_video.pid';

    // Locking mechanism to prevent multiple instances
    if (fs.existsSync(pidFile)) {
        console.error('Another instance of the script is already running.');
        process.exit(1);
    }
    fs.writeFileSync(pidFile, process.pid.toString());
    process.on('exit', () => {
        if (fs.existsSync(pidFile)) {
            fs.unlinkSync(pidFile);
        }
    });
    process.on('SIGINT', () => {
        process.exit();
    });

    console.log('Launching browser...');
    const browser = await puppeteer.launch({
        executablePath: '/usr/bin/google-chrome',
        headless: true,
        defaultViewport: { width: 720, height: 1280 },
        args: [
            '--no-sandbox',
            '--disable-setuid-sandbox',
            '--allow-file-access-from-files',
            '--disable-dev-shm-usage',
            '--disable-gpu',
            '--no-first-run',
            '--disable-notifications',
            '--mute-audio',
            '--disable-background-timer-throttling',
            '--disable-backgrounding-occluded-windows',
            '--disable-renderer-backgrounding',
        ],
    });

    const page = await browser.newPage();
    page.setDefaultTimeout(60000);

    // Monitor console logs from the page
    page.on('console', (msg) => {
        console.log(`PAGE LOG: ${msg.text()}`);
    });

    console.log('Navigating to index.html...');
    await page.goto(`file://${__dirname}/index.html`);

    console.log('Waiting for video metadata to load...');
    try {
        await page.waitForFunction(() => {
            const video = document.getElementById('video');
            return video && video.readyState >= 1;
        }, { timeout: 10000 });
        console.log('Video metadata loaded.');
    } catch (error) {
        console.error('Error waiting for video metadata:', error);
        await browser.close();
        return;
    }

    console.log('Fetching video duration...');
    const videoDuration = await page.evaluate(() => {
        const video = document.getElementById('video');
        return video.duration;
    });
    console.log(`Video duration: ${videoDuration} seconds`);

    const fps = 30;
    const totalFrames = Math.ceil(videoDuration * fps);

    console.log('Starting video recording...');

    // Extract audio from input_video.mp4
    const inputVideoPath = path.join(__dirname, 'input_video.mp4');
    const tempAudioPath = path.join(__dirname, 'temp_audio.aac');

    console.log('Extracting audio from input_video.mp4...');
    await new Promise((resolve, reject) => {
        ffmpeg(inputVideoPath)
            .noVideo()
            .audioCodec('copy')
            .output(tempAudioPath)
            .on('end', () => {
                console.log('Audio extraction complete.');
                resolve();
            })
            .on('error', (err) => {
                console.error('Error extracting audio:', err);
                reject(err);
            })
            .run();
    });

    const frameStream = new stream.PassThrough();
    const outputVideoPath = path.join(__dirname, 'output_video.mp4');
    const proc = ffmpeg()
        .input(frameStream)
        .inputFormat('image2pipe')
        .inputFPS(fps)
        .addInput(tempAudioPath) // Add the extracted audio
        .videoCodec('libx264')
        .audioCodec('aac')
        .outputOptions('-preset', 'veryfast', '-pix_fmt', 'yuv420p')
        .output(outputVideoPath)
        .on('end', () => {
            console.log('Video recording complete.');
            // Clean up temporary audio file
            fs.unlinkSync(tempAudioPath);
            browser.close();
        })
        .on('error', (err) => {
            console.error('Error during video recording:', err);
            // Clean up temporary audio file
            fs.unlinkSync(tempAudioPath);
            browser.close();
        })
        .run();

    for (let i = 0; i < totalFrames; i++) {
        const currentTime = i / fps;
        console.log(`Processing frame ${i + 1}/${totalFrames} at ${currentTime.toFixed(2)}s`);

        // Update video time
        await page.evaluate((t) => {
            const video = document.getElementById('video');
            video.currentTime = t;
        }, currentTime);

        // Wait for video to seek
        await page.evaluate(() => {
            return new Promise((resolve) => {
                const video = document.getElementById('video');
                if (video.readyState >= 2) {
                    resolve();
                } else {
                    video.addEventListener('seeked', resolve, { once: true });
                }
            });
        });

        // Capture screenshot
        const screenshot = await page.screenshot({ type: 'jpeg', quality: 80 });
        frameStream.write(screenshot);

        // Maintain frame rate
        await new Promise((resolve) => setTimeout(resolve, 1000 / fps));
    }

    frameStream.end();
})();
