(async () => {
    try {
        const response = await fetch('captions.json');
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const captionsData = await response.json();

        const video = document.getElementById('video');
        const svgCaptions = document.getElementById('svg-captions');

        const maxCharsPerLine = 12;
        const maxLines = 2;

        // Step 1: Pre-chunk the captions into manageable segments
        let chunks = [];
        let currentChunk = [];
        let currentChunkStartTime = captionsData[0].startTime;

        captionsData.forEach((wordObj, index) => {
            let chunkText = currentChunk.map(w => w.text).join(' ');

            if ((chunkText.length + wordObj.text.length + 1) <= maxCharsPerLine * maxLines) {
                currentChunk.push(wordObj);
            } else {
                // End the current chunk
                chunks.push({
                    startTime: currentChunkStartTime,
                    endTime: currentChunk[currentChunk.length - 1].endTime,
                    words: currentChunk
                });

                // Start a new chunk
                currentChunk = [wordObj];
                currentChunkStartTime = wordObj.startTime;
            }

            // Add the remaining chunk if this is the last word
            if (index === captionsData.length - 1) {
                chunks.push({
                    startTime: currentChunkStartTime,
                    endTime: currentChunk[currentChunk.length - 1].endTime,
                    words: currentChunk
                });
            }
        });

        // Variables to track state
        let currentChunkIndex = 0;
        let textElement = null;

        function updateCaptions() {
            const currentTime = video.currentTime;

            if (currentChunkIndex < chunks.length) {
                const currentChunk = chunks[currentChunkIndex];

                if (currentTime >= currentChunk.startTime && currentTime < currentChunk.endTime) {
                    if (!textElement) {
                        // Create a new text element for the chunk
                        textElement = document.createElementNS('http://www.w3.org/2000/svg', 'text');
                        textElement.setAttribute('x', '50%');
                        textElement.setAttribute('y', '85%');
                        textElement.setAttribute('dominant-baseline', 'middle');
                        textElement.setAttribute('text-anchor', 'middle');
                        textElement.setAttribute('font-size', '36px');

                        let currentLineLength = 0;
                        let lineCount = 0;
                        let lineWords = [];

                        currentChunk.words.forEach((wordObj, i) => {
                            lineWords.push(wordObj.text);
                            currentLineLength += wordObj.text.length + 1; // Add word length + 1 space

                            // If line length exceeds the max characters or it's the last word, wrap to a new line
                            if (currentLineLength >= maxCharsPerLine || i === currentChunk.words.length - 1) {
                                // Create a new tspan for this line
                                const tspan = document.createElementNS('http://www.w3.org/2000/svg', 'tspan');
                                tspan.setAttribute('x', '50%'); // Center line
                                tspan.setAttribute('dy', lineCount === 0 ? '0' : '1.2em'); // Adjust line height
                                tspan.textContent = lineWords.join(' '); // Set the line's text

                                // Append to the text element
                                textElement.appendChild(tspan);

                                // Prepare for the next line
                                lineWords = [];
                                currentLineLength = 0;
                                lineCount++;
                            }
                        });

                        svgCaptions.appendChild(textElement);
                    }

                    // Highlight words based on individual timings
                    highlightWords(currentTime, currentChunk);
                } else if (currentTime >= currentChunk.endTime) {
                    // Move to the next chunk
                    currentChunkIndex++;

                    // Remove the text element when the chunk ends
                    if (textElement) {
                        svgCaptions.removeChild(textElement);
                        textElement = null;
                    }
                }
            }

            requestAnimationFrame(updateCaptions);
        }

        // Function to highlight words within the current chunk
        function highlightWords(currentTime, currentChunk) {
            const tspans = Array.from(textElement.getElementsByTagName('tspan'));

            tspans.forEach((tspan, tspanIndex) => {
                const words = tspan.textContent.split(' ');
                words.forEach((word, index) => {
                    const wordObj = currentChunk.words.find(w => w.text === word);
                    if (wordObj && currentTime >= wordObj.startTime && currentTime < wordObj.endTime) {
                        // Highlight only the current word
                        words[index] = `<tspan class="highlight">${word}</tspan>`;
                    } else {
                        // Reset other words
                        words[index] = `<tspan>${word}</tspan>`;
                    }
                });

                tspan.innerHTML = words.join(' ');
            });
        }

        // Start updating captions
        updateCaptions();

    } catch (error) {
        console.error('Error fetching or processing captions:', error);
    }
})();
